# -*- coding: utf-8 -*-
"""
Created on Tue Nov 29 09:09:10 2016

@author: andrey
"""
from PythonQt import QtCore, QtGui
from PythonQt.Road import ObjectTreeItem
from priority.geometry import xyToGeo, geoToXY, MeterToDistance
from priority.vdetector import Vdetector
from menu import SubMenu
from gui import osmView
import RoadCentre


class RouteItem(QtGui.QGraphicsItem):
    # Ширина маршрута в метрах
    width = 2.0

    def __init__(self, layer, geometry):
        QtGui.QGraphicsItem.__init__(self)
        self.setFlags(QtGui.QGraphicsItem.ItemIsSelectable)
        self.setCursor(QtCore.Qt.ArrowCursor)
        self.bound = QtCore.QRectF()
        self.basePath = QtGui.QPainterPath()
        self.basePath.setFillRule(QtCore.Qt.WindingFill)
        self.directionArrows = QtGui.QPainterPath()
        self.setParentItem(layer)
        self.setVisible(False)
        self.color = QtCore.Qt.lightGray
        self.has_geometry = False
        self.setGeometry(geometry)

    def setGeometry(self, geometry):
        if geometry is None:
            return
        base_point = None
        self.prepareGeometryChange()
        skel = QtGui.QPainterPath()
        for point_str in geometry.split(' '):
            pp = [float(p) for p in point_str.split(',')]
            if not base_point:
                base_point = QtCore.QPointF(pp[0], pp[1])
            fp = geoToXY(QtCore.QPointF(pp[0], pp[1]))
            if skel.elementCount() == 0:
                skel.moveTo(fp)
            else:
                skel.lineTo(fp)
        stroker = QtGui.QPainterPathStroker()
        stroker.setWidth(MeterToDistance(self.width, base_point))
        # Здесь нельзя вызывать simplified() - т.к. работает непредсказуемо
        poly = stroker.createStroke(skel)
        for i in range(poly.elementCount()):
            if i:
                self.basePath.lineTo(xyToGeo(poly.elementAt(i).operator_cast_QPointF()))
            else:
                self.basePath.moveTo(xyToGeo(poly.elementAt(i).operator_cast_QPointF()))
        self.basePath.closeSubpath()
        arrow_total_length = MeterToDistance(self.width / 2.0, base_point)
        arrow_len = arrow_total_length / 1.7
        dy = arrow_len / 3.0
        step = arrow_total_length * 2.0 / skel.length()
        s = step
        while s <= 1.0:
            p1 = skel.pointAtPercent(s - step)
            p2 = skel.pointAtPercent(s)
            direction = QtCore.QLineF(p1, p2)
            t = QtGui.QTransform()
            t.translate(p2.x(), p2.y())
            t.rotate(-direction.angle())
            self.directionArrows.moveTo(xyToGeo(p1))
            self.directionArrows.lineTo(xyToGeo(p2))
            self.directionArrows.lineTo(xyToGeo(t.map(QtCore.QPointF(-arrow_len, -dy))))
            self.directionArrows.moveTo(xyToGeo(p2))
            self.directionArrows.lineTo(xyToGeo(t.map(QtCore.QPointF(-arrow_len, dy))))
            s += step * 3.0

        self.bound = self.basePath.boundingRect()
        self.has_geometry = True

    def boundingRect(self):
        return self.bound

    def shape(self):
        return self.basePath

    def paint(self, painter, opt, w):
        pen = painter.pen()
        if not pen.isCosmetic():
            pen.setCosmetic(True)
        # if self.isSelected():
        #    cur_color=QtCore.Qt.green
        # painter.setBrush(QtGui.QBrush(cur_color))
        painter.fillPath(self.basePath, QtGui.QBrush(self.color))
        painter.strokePath(self.directionArrows, pen)


class Route(ObjectTreeItem):
    def __init__(self, parent, layer, name, description, geometry, tls):
        ObjectTreeItem.__init__(self, parent)
        self.dontHide = False
        self.useSgEvents = False
        self.setProperty('locked', False)
        self.setProperty('id', name)
        self.setProperty('description', description)
        self.routeItem = RouteItem(layer, geometry)
        self.routeItem.setData(0, self)
        self.updateType()
        self.connect('selectedChanged(bool)', self.selectedChanged)
        self.setProperty("itemFlags",
                         QtCore.Qt.ItemIsSelectable | QtCore.Qt.ItemIsDropEnabled | QtCore.Qt.ItemIsEnabled)
        self.detectors = []
        self.tls = tls

    def cleanup(self):
        self.disconnect('selectedChanged(bool)', self.selectedChanged)
        self.routeItem.delete()
        self.detectors = []
        self.deleteLater()

    def setAlwaysVisible(self, v):
        if self.dontHide == v:
            return
        if v:
            self.routeItem.setVisible(True)
        self.dontHide = v
        for dt in self.detectors:
            dt.dontHide = v
            if v:
                dt.vdtItem.setVisible2(True,self)
        RoadCentre.priority.alwaysVisibleRoutes[int(self.id)] = v

    def setUseSgEvents(self, use_sg):
        if self.useSgEvents == use_sg:
            return
        self.useSgEvents = use_sg
        for tl_id in self.tls:
            light = RoadCentre.project.object(tl_id)
            if not light:
                continue
            tlo_use_sg_cnt = light.property('tlo_use_sg_cnt')
            if tlo_use_sg_cnt is None:
                tlo_use_sg_cnt = 0
            if self.useSgEvents:
                light.eventMask |= (1 << 5)

                tlo_use_sg_cnt += 1
                light.setProperty('tlo_use_sg_cnt', tlo_use_sg_cnt)
            elif light.eventMask & (1 << 5):
                if tlo_use_sg_cnt > 0:
                    tlo_use_sg_cnt -= 1
                    light.setProperty('tlo_use_sg_cnt', tlo_use_sg_cnt)
                if tlo_use_sg_cnt == 0:
                    light.eventMask &= ~(1 << 5)

        if self.useSgEvents:
            RoadCentre.priority.getDirectionsGeometry(self.tls)
        else:
            for tl_id in self.tls:
                RoadCentre.priority.resetDirectionsForObject(tl_id, True)


    def addDetector(self, dt):
        if dt not in self.detectors:
            dt.addRoute(self)
            self.detectors.append(dt)

    def removeDetector(self, dt):
        if dt in self.detectors:
            self.detectors.remove(dt)

    def updateType(self):
        tooltip_str = type_str = u'Маршрут'
        if self.property("locked"):
            type_str += u' (Заблокирован)'
        self.setProperty('type', type_str)
        desc = self.property('description')
        if desc is None:
            desc = self.id
        tooltip_str += u' ' + desc
        self.routeItem.setToolTip(tooltip_str)

    def setLocked(self, lock_state):
        if self.property("locked") == lock_state:
            return
        self.setProperty('locked', lock_state)
        self.updateType()
        self.routeItem.color = QtCore.Qt.red if lock_state else QtCore.Qt.lightGray
        self.routeItem.update()

    def loadGeometry(self):
        if not self.routeItem.has_geometry:
            RoadCentre.priority.getRouteGeometry(int(self.id))

    def loadDetectorsGeometry(self):
        for dt in self.detectors:
            if not dt.vdtItem.has_geometry:
                RoadCentre.priority.getVdtGeometry(dt.enterId, dt.leaveId)

    def selectedChanged(self, s):
        if not s and self.dontHide:
            return
        self.routeItem.setVisible(s)
        self.routeItem.setSelected(s)
        if s and not self.routeItem.has_geometry:
            RoadCentre.priority.getRouteGeometry(self.property("id"))
        for dt in self.detectors:
            if dt.vdtItem.isSelected() and not s:
                continue
            if s and not dt.vdtItem.has_geometry:
                RoadCentre.priority.getVdtGeometry(dt.enterId, dt.leaveId)
            dt.vdtItem.setVisible2(s,self)


class RouteMenu(SubMenu):
    def __init__(self):
        SubMenu.__init__(self, (u'Маршрут', u'Маршруты'))
        a = self.menuMain.addAction(u'Показать на карте')
        a.connect('triggered()', self.showOnMap)
        if hasattr(RoadCentre.plugins, "TimelinePlugin"):
            a = self.menuMain.addAction(u'Лента времени')
            a.connect('triggered()', self.showTimeline)
            a = self.menuMain.addAction(u'Лента времени (архив)')
            a.connect('triggered()', self.showArchiveTimeline)
        self.alwaysVisibleAction = self.menuMain.addAction(u'Всегда видимый')
        self.alwaysVisibleAction.checkable = True
        self.alwaysVisibleAction.connect('triggered(bool)', self.alwaysVisible)
        self.useSignalGroupsEvents = self.menuMain.addAction(u'Использовать события от СГ')
        self.useSignalGroupsEvents.checkable = True
        self.useSignalGroupsEvents.connect('triggered(bool)', self.useSGEvents)

        self.selection = None

    def selectedRoutes(self):
        for o in RoadCentre.kernel.selection():
            if hasattr(o, "className") and o.className() == "Route":
                yield o

    def menu(self, obj):
        m = SubMenu.menu(self, obj)
        vis_act = False
        use_sg_act = False
        lock_act = None
        for r in self.selectedRoutes():
            if not vis_act:
                self.alwaysVisibleAction.checked = r.dontHide
                vis_act = True
            if not use_sg_act:
                self.useSignalGroupsEvents.checked = r.useSgEvents
                use_sg_act = True
            if lock_act is None:
                lock_state = r.property("locked")
                if lock_state is not None:
                    lock_act = m.addAction(u'Разблокировать' if lock_state else u'Заблокирвать')
                    lock_act.connect('triggered()', self.unlockRoute if lock_state else self.lockRoute)
        return m

    def showOnMap(self):
        for r in self.selectedRoutes():
            r.selected = True
            osmView.fitItem(r.routeItem)

    def alwaysVisible(self, checked):
        for r in self.selectedRoutes():
            r.setAlwaysVisible(checked)

    def lockRoute(self):
        for r in self.selectedRoutes():
            RoadCentre.priority.lockRoute(r.id)

    def unlockRoute(self):
        for r in self.selectedRoutes():
            RoadCentre.priority.unlockRoute(r.id)

    def showTimeline(self):
        for r in self.selectedRoutes():
            RoadCentre.priority.showTimeline(r.id)
            break

    def useSGEvents(self, checked):
        for r in self.selectedRoutes():
            r.setUseSgEvents(checked)

    def showArchiveTimeline(self):
        for r in self.selectedRoutes():
            RoadCentre.priority.showTimeline(r.id, True)
            break
