# -*- coding: utf-8 -*-
"""
(c)2016 ЗАО "РИПАС" www.ripas.ru
"""

from PythonQt import QtGui, QtCore
from PythonQt.Road import ObjectTreeItem
import RoadCentre
from menu import SubMenu
from gui import osmView, mainWindow
from priority.route import Route
from priority.geometry import geoToXY, xyToGeo, MeterToDistance
import math
import datetime
import re

class VehicleItem(QtGui.QGraphicsObject):
    BUS = 1
    POI = 2
    MAX_ZOOM = 17
    MIN_ZOOM = 10
    POI_SCALE = 1.0
    UPDATE_EVID = None

    def __init__(self, pt_pos, layer, cur_zoom):
        QtGui.QGraphicsObject.__init__(self, layer)
        self.bound = QtCore.QRectF()
        self.setFlags(QtGui.QGraphicsItem.ItemIsSelectable)
        self.setCursor(QtCore.Qt.ArrowCursor)
        self.setPos(pt_pos)
        self.lastZoom = cur_zoom
        self.angle = 0.0
        self.fillColor = QtCore.Qt.lightGray
        self.viewRadius = None
        self.viewAngle = None
        self.sectorVisible = False
        self.viewType = self.POI
        self.poiShape = QtGui.QPainterPath()
        self.busShape = QtGui.QPainterPath()
        self.view_sector = QtGui.QPainterPath()
        self.vehNum = None
        self.showBoardNumber = False
        if VehicleItem.UPDATE_EVID is None:
            VehicleItem.UPDATE_EVID = QtCore.QEvent.registerEventType()
        self.updateShape()

    def event(self, event):
        if event.type() == self.UPDATE_EVID:
            event.accept()
            self.__updateShape__()
            return True
        return False

    def updateShape(self):
        QtCore.QCoreApplication.postEvent(self,QtCore.QEvent(VehicleItem.UPDATE_EVID))

    def __updateShape__(self):
        pt_pos = self.pos
        pt_xy_pos = geoToXY(pt_pos)
        self.viewType = self.POI
        # s2z=lambda z:-0.3*z*z+3.5*z+25
        s2z = lambda z: 24295 * math.exp(-0.55 * z)
        k = 1
        if self.lastZoom >= VehicleItem.MAX_ZOOM:
            self.viewType = self.BUS
        elif self.lastZoom > VehicleItem.MIN_ZOOM:
            k = s2z(self.lastZoom)
        else:
            k = s2z(VehicleItem.MIN_ZOOM)
        k *= VehicleItem.POI_SCALE
        mp = lambda x, y: QtCore.QPointF(MeterToDistance(x * k, pt_pos), MeterToDistance(y * k, pt_pos))
        md = lambda m: MeterToDistance(m * k, pt_pos)

        if self.viewType == self.POI:
            poi_shape = QtGui.QPainterPath()
            poi_shape.setFillRule(QtCore.Qt.WindingFill)
            poi_shape.moveTo(mp(0, 7.7231))
            poi_shape.arcTo(md(-15.4057), md(2.3174), md(15.8114), md(15.8114), 18, 36)
            poi_shape.arcTo(md(-5), md(-5.2769), md(10), md(10), 236, -292)
            poi_shape.arcTo(md(-0.4057), md(2.3174), md(15.8114), md(15.8114), 126, 36)
            self.bound = poi_shape.boundingRect()
            ts = min(self.bound.width(),self.bound.height())
            t = QtGui.QTransform()
            t.rotate(self.angle + 180)
            poi_shape = t.map(poi_shape)
            poi_shape.translate(pt_xy_pos)
            for i in range(poi_shape.elementCount()):
                geo_pos = xyToGeo(poi_shape.elementAt(i).operator_cast_QPointF())
                poi_shape.setElementPositionAt(i, geo_pos.x(), geo_pos.y())
            # translate(-pt_pos) вызывает падение - какой-то косяк в pythonqt
            poi_shape.translate(QtCore.QPointF() - pt_pos)
            self.prepareGeometryChange()
            self.bound = poi_shape.boundingRect()
            self.poiShape = poi_shape
        else:
            bus_shape = QtGui.QPainterPath()
            bus_shape.setFillRule(QtCore.Qt.WindingFill)
            bus_shape.addRect(md(-1.25), md(-4), md(2.5), md(10))
            bus_shape.addRect(md(-0.5), md(-2.5), md(1), md(1))
            bus_shape.addRect(md(-0.5), md(2), md(1), md(1))
            bus_shape.moveTo(mp(-1.25, -4))
            bus_shape.lineTo(mp(-1.25, -5.5))
            bus_shape.lineTo(mp(-0.75, -6))
            bus_shape.lineTo(mp(0.75, -6))
            bus_shape.lineTo(mp(1.25, -5.5))
            bus_shape.lineTo(mp(1.25, -4))
            bus_shape.closeSubpath()
            bus_shape.moveTo(mp(-0.75, -6))
            bus_shape.lineTo(mp(-0.5, -4.5))
            bus_shape.lineTo(mp(0.5, -4.5))
            bus_shape.lineTo(mp(0.75, -6))
            bus_shape.closeSubpath()
            t = QtGui.QTransform()
            t.rotate(self.angle)
            bus_shape = t.map(bus_shape)
            bus_shape.translate(pt_xy_pos)
            for i in range(bus_shape.elementCount()):
                geo_pos = xyToGeo(bus_shape.elementAt(i).operator_cast_QPointF())
                bus_shape.setElementPositionAt(i, geo_pos.x(), geo_pos.y())
            # translate(-pt_pos) вызывает падение - какой-то косяк в pythonqt
            bus_shape.translate(QtCore.QPointF() - pt_pos)
            self.prepareGeometryChange()
            self.bound = bus_shape.boundingRect()
            self.busShape = bus_shape

        self.view_sector = QtGui.QPainterPath()
        if self.viewRadius is not None and self.viewAngle is not None:
            k = 1.0
            view_diam = md(self.viewRadius) * 2.0
            view_angle = self.viewAngle
            r = QtCore.QRectF(0, 0, view_diam, view_diam)
            r.moveCenter(mp(0, 0))
            view_sector = QtGui.QPainterPath()
            view_sector.moveTo(mp(0,0))
            view_sector.arcTo(r, 90 - self.angle - view_angle/2.0, view_angle)
            view_sector.closeSubpath()
            view_sector.translate(pt_xy_pos)
            for i in range(view_sector.elementCount()):
                geo_pos = xyToGeo(view_sector.elementAt(i).operator_cast_QPointF())
                view_sector.setElementPositionAt(i, geo_pos.x(), geo_pos.y())
            view_sector.translate(QtCore.QPointF() - pt_pos)
            self.prepareGeometryChange()
            self.bound = self.bound.united(view_sector.boundingRect())
            self.view_sector = view_sector
        #QtCore.QCoreApplication.processEvents()

    def boundingRect(self):
        return self.bound

    def rotate(self, ang):
        if ang == self.angle:
            return
        self.angle = ang
        self.updateShape()

    def setFillColor(self, clr):
        if self.fillColor == clr:
            return
        self.fillColor = clr
        self.update()

    def shape(self):
        return self.poiShape if self.viewType == self.POI else self.busShape

    def paint(self, painter, opt, w):
        pen = painter.pen()
        if not pen.isCosmetic():
            pen.setCosmetic(True)
        painter.setBrush(QtGui.QBrush(self.fillColor))
        if self.isSelected():
            pen.setStyle(QtCore.Qt.SolidLine)
            pen.setWidth(2)
            pen.setCosmetic(True)
        painter.drawPath(self.shape())
        if self.sectorVisible:
            painter.setBrush(QtGui.QBrush(QtCore.Qt.blue))
            painter.setOpacity(0.5)
            painter.drawPath(self.view_sector)
        if self.showBoardNumber and self.vehNum is not None and self.viewType == self.POI:
            pen.setWidth(2)
            pen.setColor(QtCore.Qt.black)
            painter.setPen(pen)
            t = painter.transform()
            t.scale(1/t.m11(),1/t.m22())
            rc = t.mapRect(QtCore.QRectF(-8,-8,16,16))
            painter.resetTransform()
            f = painter.font()
            if self.lastZoom <= 13:
                f.setPointSize(f.pointSize()/1.6)
            f.setBold(True)
            painter.setFont(f)
            painter.drawText(rc,QtCore.Qt.AlignCenter, self.vehNum)


class Vehicle(ObjectTreeItem):
    TEXT_SHOW_NONE = 1
    TEXT_SHOW_SELECTED = 2
    TEXT_SHOW_ALWAYS = 3

    def __init__(self, parent, pos, name=None, description=None):
        ObjectTreeItem.__init__(self)
        self.priorityState = False
        self.showSnapRadius = False
        self.trackMovement = False
        self.snapRadius = None
        self.snapRadiusItem = None
        self.real_pos = None
        self.textShowMode = Vehicle.TEXT_SHOW_NONE
        self.sectorShowMode = Vehicle.TEXT_SHOW_NONE
        self.iconItem = VehicleItem(QtCore.QPointF(), parent, RoadCentre.gui.osmView.zoom)
        self.hasGPSData = (pos is not None)
        if self.hasGPSData:
            self.iconItem.setPos(*pos)
        self.iconItem.setVisible(False)
        self.iconItem.setData(0, self)
        self.iconItem.setZValue(0.9)
        self.captionItem = None
        self.cur_speed = None
        self.hasRoute = False
        self.onRoute = False
        if name is not None:
            m = re.match(r'.*?(\d+)$',name)
            if m is not None:
                self.iconItem.vehNum = m.group(1)

        self.name = name
        self.description = description
        self.setProperty('id', self.name)
        self.setProperty('description', self.description)
        self.setProperty('type', u'Борт')
        self.connect('selectedChanged(bool)', self.selectedChanged)
        self.lastConnectedTime = None
        self.updateToolTip()
        self.updateIcon()
        self.setProperty("itemFlags",
                         QtCore.Qt.ItemIsSelectable | QtCore.Qt.ItemIsDragEnabled | QtCore.Qt.ItemIsEnabled)
        self.activatedDetectors = []

    def __del__(self):
        self.iconItem.delete()

    def cleanup(self):
        self.iconItem.setVisible(False)
        if self.real_pos:
            self.real_pos.delete()
            self.real_pos = None
        if self.snapRadiusItem:
            self.snapRadiusItem.delete()
            self.snapRadiusItem = None
        if self.captionItem:
            self.captionItem.setVisible(False)
        self.lastConnectedTime = datetime.datetime.now()
        self.cur_cpeed = None
        self.hasGPSData = False
        self.onRoute = False
        self.hasRoute = False
        self.updateToolTip()
        self.updateIcon()

    def selectedChanged(self, s):
        self.iconItem.setSelected(s)
        if self.real_pos:
            self.real_pos.setVisible(s)
        self.updateCaptionVisibility()
        self.updateEmergencySectorVisibility()

    def updateEmergencySectorVisibility(self):
        if not self.iconItem:
            return
        vis = False
        if bool(self.property("allow_emergency")):
            vis = self.iconItem.isSelected()
            if self.sectorShowMode == Vehicle.TEXT_SHOW_ALWAYS:
                vis = True
            elif self.sectorShowMode == Vehicle.TEXT_SHOW_NONE:
                vis = False
            if not self.iconItem.isVisible():
                vis = False
            if vis == self.iconItem.sectorVisible:
                return
        self.iconItem.sectorVisible = vis
        self.iconItem.update()

    def updateCaptionVisibility(self):
        if not self.captionItem:
            return
        vis = self.iconItem.isSelected()
        if self.textShowMode == Vehicle.TEXT_SHOW_ALWAYS:
            vis = True
        elif self.textShowMode == Vehicle.TEXT_SHOW_NONE:
            vis = False
        if not self.iconItem.isVisible():
            vis = False
        if vis == self.captionItem.isVisible():
            return
        self.captionItem.setVisible(vis)

    def setPriorityState(self, s):
        if self.priorityState == s:
            return
        self.priorityState = s
        self.updateToolTip()
        self.updateIcon()

    def changePOIScale(self, scale):
        if self.iconItem.poiScale == scale:
            return
        self.iconItem.poiScale = scale
        self.iconItem.updateShape()

    def updateIcon(self):
        base_icon_name = "bus-24"
        is_emergency = bool(self.property("allow_emergency"))
        if is_emergency:
            base_icon_name = "ecar"
        if self.property("locked"):
            self.setProperty('icon', QtGui.QPixmap("images/%sr.png"%base_icon_name))
            self.iconItem.setFillColor(QtCore.Qt.red)
        elif self.hasGPSData:
            if self.priorityState:
                self.setProperty('icon', QtGui.QPixmap("images/%sp.png"%base_icon_name))
                self.iconItem.setFillColor(QtCore.Qt.green)
            else:
                self.setProperty('icon', QtGui.QPixmap("images/%s.png"%base_icon_name))
                if self.hasRoute:
                    if self.onRoute:
                        self.iconItem.setFillColor(QtGui.QColor(0xc7e2f8))
                    else:
                        self.iconItem.setFillColor(QtGui.QColor(0xffb6c1))
                else:
                    self.iconItem.setFillColor(QtCore.Qt.lightGray)
        else:
            self.setProperty('icon', QtGui.QPixmap("images/%snogps.png"%base_icon_name))
            self.iconItem.setFillColor(QtCore.Qt.lightGray)
        RoadCentre.gui.treeView.model().layoutChanged.emit()

    def setSnapRadiusVisible(self, s):
        if self.showSnapRadius == s:
            return
        self.showSnapRadius = s
        if self.snapRadiusItem:
            if not self.showSnapRadius:
                self.snapRadiusItem.delete()
                self.snapRadiusItem = None
            else:
                self.snapRadiusItem.setVisible(self.showSnapRadius)
                self.snapRadiusItem.update()

    def enableCaptionItem(self, en):
        if (self.captionItem is not None) == en:
            return
        if self.captionItem:
            self.captionItem.delete()
            self.captionItem = None
            return
        self.captionItem = RoadCentre.kernel.graphicsScene.addText("")
        self.captionItem.setDefaultTextColor(QtCore.Qt.cyan)
        self.captionItem.setParentItem(self.iconItem.parentItem())
        self.captionItem.setFlags(QtGui.QGraphicsItem.ItemIgnoresTransformations)
        self.captionItem.setZValue(1.1)
        fnt = self.captionItem.font()
        fnt.setBold(True)
        fnt.setWeight(QtGui.QFont.Black)
        fnt.setPointSize(10)
        self.captionItem.setFont(fnt)
        cur = self.captionItem.textCursor()
        fmt = cur.charFormat()
        pen = QtGui.QPen()
        pen.setCosmetic(True)
        pen.setColor(QtCore.Qt.black)
        fmt.setTextOutline(pen)
        cur.setCharFormat(fmt)
        self.captionItem.setTextCursor(cur)
        self.updateToolTip()

    # rpos - спроецированная на маршрут позиция, pos - реальная позиция
    def move(self, rpos, pos, on_route):
        has_gps = (rpos is not None and pos is not None)
        update_tool_tip = True
        update_icon = False
        if self.lastConnectedTime is not None:
            self.lastConnectedTime = None
            update_tool_tip = True
        if has_gps != self.hasGPSData:
            self.hasGPSData = has_gps
            update_tool_tip = True
            update_icon = True
        if update_tool_tip:
            self.updateToolTip()
        if on_route != self.onRoute:
            self.onRoute = on_route
            update_icon = True
        self.updateCaptionVisibility()
        self.updateEmergencySectorVisibility()
        if self.hasGPSData:
            if not self.iconItem.isVisible():
                self.iconItem.setVisible(True)
                if self.real_pos:
                    self.real_pos.setVisible(self.iconItem.isSelected())
                if self.snapRadiusItem:
                    self.snapRadiusItem.setVisible(True)
                update_icon = True
            pt_pos = QtCore.QPointF(*pos)
            pt_rpos = QtCore.QPointF(*rpos)
            if self.captionItem:
                self.captionItem.setPos(pt_rpos + QtCore.QPointF(0.0002, 0))
            self.iconItem.setPos(pt_rpos)
            if self.showSnapRadius and self.snapRadius:
                if not self.snapRadiusItem:
                    self.snapRadiusItem = RoadCentre.kernel.graphicsScene.addEllipse(QtCore.QRectF())
                    self.snapRadiusItem.setZValue(1.0)
                    self.snapRadiusItem.setParentItem(self.iconItem.parentItem())
                    self.snapRadiusItem.setVisible(self.iconItem.isVisible())
                xy_pos = geoToXY(pt_pos)
                d = MeterToDistance(self.snapRadius * 2, pt_pos)
                r = QtCore.QRectF(-d / 2.0, -d / 2.0, d, d)
                r.moveCenter(xy_pos)
                r1 = QtCore.QRectF()
                r1.setTopLeft(xyToGeo(r.topLeft()))
                r1.setBottomRight(xyToGeo(r.bottomRight()))
                self.snapRadiusItem.setRect(r1)
            if pos is not None:
                if not self.real_pos:
                    pt_size = 0.00002
                    self.real_pos = RoadCentre.kernel.graphicsScene.addEllipse(-pt_size / 2.0, -pt_size / 4.0, pt_size,
                                                                               pt_size / 2.0)
                    self.real_pos.setParentItem(self.iconItem.parentItem())
                    if pos is not None:
                        self.real_pos.setPos(pt_pos)
                    self.real_pos.setBrush(QtGui.QBrush(QtCore.Qt.darkGreen))
                    self.real_pos.setZValue(1)
                    self.real_pos.setVisible(self.iconItem.isSelected())
                self.real_pos.setPos(*pos)
        else:
            if self.iconItem.isVisible():
                self.iconItem.setVisible(False)
                if self.real_pos:
                    self.real_pos.setVisible(False)
                if self.snapRadiusItem:
                    self.snapRadiusItem.setVisible(False)
                update_icon = True
        if update_icon:
            self.updateIcon()
        if self.trackMovement and self.hasGPSData:
            osmView.centerOn(self.iconItem)

    def setSnapRadius(self, radius):
        if self.snapRadius == radius:
            return
        self.snapRadius = radius

    def rotate(self, ang):
        self.iconItem.rotate(ang)

    def setEmergencyData(self, radius, sweep):
        if self.iconItem.viewRadius != radius or self.iconItem.viewAngle != sweep:
            self.iconItem.viewRadius = radius
            self.iconItem.viewAngle = sweep
            self.iconItem.updateShape()

    def setSpeed(self, spd):
        if self.cur_speed == spd:
            return
        self.cur_speed = spd
        self.updateToolTip()

    def updateToolTip(self):
        base_name = u'Борт'
        if bool(self.property("allow_emergency")):
            base_name = u'МЭС'
        toolTip = base_name + ' ' + (self.name if self.name else '?')
        typeStr = base_name
        descr = self.property("description")
        if descr:
            toolTip += u'\n' + descr
        if self.cur_speed is not None:
            toolTip += u'\nСкорость: ' + str(self.cur_speed) + u'км/ч'
        if not self.hasGPSData:
            toolTip += u'\nНет данных о позиции'
            typeStr += u'(Нет данных о позиции)'
        if self.property("locked"):
            toolTip += u'\nЗаблокирован'
            typeStr += u'(Заблокирован)'
        elif self.priorityState:
            toolTip += u'\nПредоставлен приоритет'
            typeStr += u'(Приоритет)'

        self.hasRoute = False
        try:
            routeItem = self.parent().parent()
            if isinstance(routeItem, Route):
                self.hasRoute = True
                toolTip += u'\nМаршрут'
                descr = routeItem.property("description")
                if descr:
                    toolTip += u' %s' % descr
                else:
                    toolTip += u' ' + routeItem.id
        except:
            pass

        if self.lastConnectedTime is not None:
            toolTip += u'\n Последний раз был на связи ' + str(self.lastConnectedTime)

        if self.captionItem:
            self.captionItem.setPlainText(toolTip)

        self.setProperty('type', typeStr)
        self.iconItem.setToolTip(toolTip)
        self.setProperty('toolTip', toolTip)

    def changeTextShowMode(self, mode):
        if self.textShowMode == mode:
            return
        self.textShowMode = mode
        if self.textShowMode == Vehicle.TEXT_SHOW_ALWAYS:
            self.enableCaptionItem(True)
            if self.captionItem:
                self.captionItem.setVisible(True)
        elif self.textShowMode == Vehicle.TEXT_SHOW_SELECTED:
            self.enableCaptionItem(True)
            if self.captionItem:
                self.captionItem.setVisible(self.iconItem.isSelected() and self.hasGPSData)
        else:
            self.enableCaptionItem(False)

    def changeSectorShowMode(self, mode):
        if self.sectorShowMode == mode:
            return
        self.sectorShowMode = mode
        self.updateEmergencySectorVisibility()

    def setShowBoardNumber(self,s):
        if self.iconItem.showBoardNumber == s:
            return
        self.iconItem.showBoardNumber = s
        self.iconItem.update()

    def set_vehicle_tracking(self,track):
        self.trackMovement = track
        if track:
            osmView.zoom = 16
            osmView.centerOn(self.iconItem)
    

class VehicleMenu(SubMenu):
    def __init__(self):
        SubMenu.__init__(self, (u'Транспортное средство', u'Транспортные средства'))
        self.routeItems = []
        a = self.menuMain.addAction(u'Посмотреть трек за сегодня')
        a.connect('triggered()', self.showTrackForToday)
        a = self.menuMain.addAction(u'Посмотреть трек за ...')
        a.connect('triggered()', self.showTrackFor)

    def selectedVehicles(self):
        for o in RoadCentre.kernel.selection():
            if hasattr(o, "className") and o.className() == "Vehicle":
                yield o

    def menu(self, obj):
        m = SubMenu.menu(self, obj)
        self.routeItems = []

        routes_menu = QtGui.QMenu(u'Назначить маршруту ...', m)
        gr = QtGui.QActionGroup(m)
        for route_id in RoadCentre.priority.routes:
            route = RoadCentre.priority.routes[route_id]
            m_title = route.property("description")
            if not m_title:
                m_title = str(route_id)
            act = routes_menu.addAction(m_title)
            act.setData(route_id)
            gr.addAction(act)
        gr.connect('triggered(QAction*)', self.assignToRoute)

        show_route_act = None
        show_act = None
        lock_act = None
        assign_act = None
        for veh in self.selectedVehicles():
            route_item = veh.parent().parent()
            if isinstance(route_item, Route):
                if route_item not in self.routeItems:
                    self.routeItems.append(route_item)
                if show_route_act is None:
                    show_route_act = m.addAction(u'Показать маршрут')
                    show_route_act.connect('triggered()', self.showRoute)
                    act = m.addAction(u'Снять с маршрута')
                    act.connect('triggered()', self.unassignFromRoute)
            if show_act is None and veh.iconItem.isVisible():
                show_act = m.addAction(u'Показать на карте')
                show_act.connect('triggered()', self.showOnMap)
            if lock_act is None:
                locked = veh.property("locked")
                if locked is None: locked = False
                lock_act = m.addAction(u'Разблокировать' if locked else u'Заблокировать')
                lock_act.connect('triggered()', self.unlockBoard if locked else self.lockBoard)
            if assign_act is None and not routes_menu.isEmpty():
                assign_act = m.addMenu(routes_menu)

        act = m.addAction(u"Свойства")
        act.connect('triggered()', self.vehProperties)
        if len(RoadCentre.kernel.selection())==1 and veh.hasGPSData:
            act = m.addAction("Отслеживать перемещения")
            act.checkable = True
            act.checked = veh.trackMovement
            act.connect('triggered(bool)', self.trackMovement)
        return m

    def showOnMap(self):
        for veh in self.selectedVehicles():
            osmView.fitItem(veh.iconItem)

    def lockBoard(self):
        for veh in self.selectedVehicles():
            RoadCentre.priority.lockBoard(veh.name)

    def unlockBoard(self):
        for veh in self.selectedVehicles():
            RoadCentre.priority.unlockBoard(veh.name)

    def showRoute(self):
        for route_item in self.routeItems:
            route_item.selected = True

    def unassignFromRoute(self):
        for veh in self.selectedVehicles():
            RoadCentre.priority.unassignBoard(veh.name)

    def assignToRoute(self, act):
        route_id = act.data()
        if route_id is None:
            return
        for veh in self.selectedVehicles():
            RoadCentre.priority.assignBoard(veh.name, route_id)

    def showTrackForToday(self):
        cur_date = QtCore.QDate.currentDate().toString(QtCore.Qt.ISODate)
        for veh in self.selectedVehicles():
            RoadCentre.priority.openTrackUrl(veh.name, cur_date)

    def showTrackFor(self):
        dlg = QtGui.QDialog(mainWindow)
        dlg.setLayout(QtGui.QVBoxLayout())
        calendar = QtGui.QCalendarWidget(dlg)
        dlg.layout().addWidget(calendar)
        btn_box = QtGui.QDialogButtonBox(dlg)
        btn_box.setOrientation(QtCore.Qt.Horizontal)
        btn_box.setStandardButtons(QtGui.QDialogButtonBox.Ok | QtGui.QDialogButtonBox.Cancel)
        btn_box.connect("accepted()", dlg.accept)
        btn_box.connect("rejected()", dlg.reject)
        dlg.layout().addWidget(btn_box)
        if getattr(dlg, "exec")() == QtGui.QDialog.Accepted:
            date_str = calendar.selectedDate.toString(QtCore.Qt.ISODate)
            for veh in self.selectedVehicles():
                RoadCentre.priority.openTrackUrl(veh.name, date_str)

    def vehProperties(self):
        RoadCentre.priority.editVehicles(list(self.selectedVehicles()))

    def trackMovement(self,checked):
        for veh in self.selectedVehicles():
            RoadCentre.priority.set_vehicle_tracking(veh,checked)
            break
