# -*- coding: utf-8 -*-
"""
Created on Wed Sep  3 12:17:27 2014

@author: andrey
"""

import json
from PythonQt import QtNetwork, QtCore, QtGui, private
from PythonQt.Road import ObjectTreeItem
from priority.vehicle import Vehicle, VehicleItem
from priority.vdetector import Vdetector
from priority.route import Route
from gui import msgError, msgDebug, widgetFromUi, mainWindow
try:
    from timeline import findSynchronizerForTimeline
except:
    findSynchronizerForTimeline = None
import time
import RoadCentre
from settings import userSettings
import webbrowser
from threading import Thread
from priority.granule import Granule
from RoadCentre import project
import re

if hasattr(RoadCentre.plugins, "TimelinePlugin"):
    from timeline import docTimelineWidget

if hasattr(QtCore, 'QObject'):
    from PythonQt.QtCore import QObject
elif hasattr(private, 'QObject'):
    QObject = private.QObject

debugMode = False
otreeSectionName = u'Приоритетный пропуск'


def setDebug(d):
    global debugMode
    debugMode = d


def debugPrint(msg):
    global debugMode
    if not debugMode:
        return
    msgDebug(msg)


class PriorityClient(QObject):
    MAX_SERVICE_TIMEOUT = 10000
    BOARDS_BRANCH_ID = u"Борты"

    def __init__(self, connection):
        QObject.__init__(self)
        self.running = False
        self.layer = RoadCentre.kernel.graphicsScene.addLayer(otreeSectionName)
        self.socket = QtNetwork.QTcpSocket(self)
        #self.socket.setProxy(QtNetwork.QNetworkProxy(QtNetwork.QNetworkProxy.NoProxy))
        self.connection = connection
        self.vehNameFilter = None
        self.routeNameFilter = None
        self.invokeId = 0
        self.getallId = None
        self.getconfigdataId = None
        self.pingId = None
        self.subscribeId = None
        self.tracked_vehicle = None
        self.getVdtId = []
        self.getRouteGeometryId = []
        self.getDirectionsGeometryId = []
        self.rpcAnswers = {}
        self.rcvbuf = ''
        self.snapRadius = None
        self.showBoardNumber = False
        self.enabled = int(userSettings.value('Priority/Enabled', 1)) != 0
        self.textShowMode = int(userSettings.value("Priority/text_show_value", Vehicle.TEXT_SHOW_SELECTED))
        self.emegrencyVehSectorShowMode = int(userSettings.value("Priority/ems_show_value", Vehicle.TEXT_SHOW_SELECTED))
        VehicleItem.MAX_ZOOM = int(userSettings.value("Priority/max_zoom", VehicleItem.MAX_ZOOM))
        VehicleItem.MIN_ZOOM = int(userSettings.value("Priority/min_zoom", VehicleItem.MIN_ZOOM))
        VehicleItem.POI_SCALE = float(userSettings.value("Priority/pict_scale", VehicleItem.POI_SCALE))

        self.reconnTimer = QtCore.QTimer(self)
        self.reconnTimer.setInterval(10000)
        self.reconnTimer.timeout.connect(self.startConnection)
        self.reconnTimer.setSingleShot(True)

        self.servicePingTimer = QtCore.QTimer(self)
        self.servicePingTimer.timeout.connect(self.servicePingTimeout)

        self.vehicles = {}
        self.boardData = {}
        self.vdetectors = {}
        self.routes = {}
        self.tl2vdt = {}
        self.priorityDefaults = {}
        self.alwaysVisibleRoutes = {}
        self.granules = {}
        self.allowedTl = set()

        self.unassignedBoardsItem = ObjectTreeItem(self)
        self.unassignedBoardsItem.setProperty("id", u"Не назначенные борты")
        self.unassignedBoardsItem.setProperty("itemFlags", QtCore.Qt.ItemIsDropEnabled | QtCore.Qt.ItemIsEnabled)
        self.assignedBoardsItem = ObjectTreeItem(self)
        self.assignedBoardsItem.setProperty("id", u"Назначенные борты")
        self.assignedBoardsItem.setProperty("itemFlags", QtCore.Qt.ItemIsEnabled)
        self.routesItem = ObjectTreeItem(self)
        self.routesItem.setProperty("id", u"Маршруты")
        self.emergencyDetectorsItem = ObjectTreeItem(self)
        self.emergencyDetectorsItem.setProperty("id", u"Детекторы экстренного вызова")
        RoadCentre.kernel.itemModel.addObjects(otreeSectionName,
                                               (self.routesItem, self.unassignedBoardsItem, self.assignedBoardsItem,
                                                self.emergencyDetectorsItem))

        self.timelineWidgets = []
        self.on_vdt_event = None
        self.lastZoom = RoadCentre.gui.osmView.zoom
        self.zoomRequest = None

        debugPrint('object PriorityClient created')

    def __del__(self):
        debugPrint('object PriorityClient deleted')

    def on_project_connectionStateChanged(self,state):
        connection = project.connection
        if state == connection.ReadyState:
            self.start()
        elif state == connection.UnconnectedState:
            self.stop()

    def start(self):
        if not self.enabled or self.running:
            return
        self.running = False
        debugPrint('starting PriorityClient')
        self.vehNameFilter = userSettings.value('Priority/VehNameFilter', None)
        if self.vehNameFilter == '':
            self.vehNameFilter = None

        self.routeNameFilter = userSettings.value('Priority/RouteNameFilter', None)
        if self.routeNameFilter == '':
            self.routeNameFilter = None

        self.showBoardNumber = int(userSettings.value('Priority/ShowBoardNumber', 0)) == 1

        self.socket.connect(QtCore.SIGNAL('readyRead()'), self.socketReadyRead)
        self.socket.connect(QtCore.SIGNAL('stateChanged(QAbstractSocket::SocketState)'), self.socketStateChanged)
        RoadCentre.kernel.itemModel.connect(QtCore.SIGNAL('dropComplete(QByteArray,QModelIndex)'),
                                            self.itemDropComplete)
        RoadCentre.kernel.itemModel.connect(QtCore.SIGNAL('itemChangeRequest(QModelIndex,QVariant,int)'),
                                            self.modelDataChanged)
        RoadCentre.project.objectNotification.connect(self.objectNotification)
        RoadCentre.project.objectConnected.connect(self.objectConnected)
        RoadCentre.gui.osmView.zoomChanged.connect(self.sceneZoomChanged)
        self.startConnection()
        self.running = True

    def stop(self):
        debugPrint('stopping PriorityClient')
        self.running = False
        self.reconnTimer.stop()
        self.servicePingTimer.stop()
        RoadCentre.gui.osmView.zoomChanged.disconnect(self.sceneZoomChanged)
        RoadCentre.project.objectNotification.disconnect(self.objectNotification)
        RoadCentre.project.objectConnected.disconnect(self.objectConnected)
        RoadCentre.kernel.itemModel.disconnect(QtCore.SIGNAL('dropComplete(QByteArray,QModelIndex)'),
                                               self.itemDropComplete)
        RoadCentre.kernel.itemModel.disconnect(QtCore.SIGNAL('itemChangeRequest(QModelIndex,QVariant,int)'),
                                               self.modelDataChanged)
        self.socket.disconnect(QtCore.SIGNAL('readyRead()'), self.socketReadyRead)
        self.socket.disconnect(QtCore.SIGNAL('stateChanged(QAbstractSocket::SocketState)'), self.socketStateChanged)
        self.socket.disconnectFromHost()
        self.socket.close()
        self.clearItems()
        self.clear_rpc_id()
        self.allowedTl.clear()

    def addTimelineWidget(self, w):
        if w not in self.timelineWidgets:
            self.timelineWidgets.append(w)

    def removeTimelineWidget(self, w):
        if w in self.timelineWidgets:
            self.timelineWidgets.remove(w)

    def clearItems(self):
        for vdt_id in self.vdetectors:
            self.vdetectors[vdt_id].cleanup()
        self.vdetectors.clear()
        for v_name in self.vehicles:
            self.vehicles[v_name].cleanup()
            self.vehicles[v_name].deleteLater()
        self.vehicles.clear()
        for route_id in self.routes:
            self.routes[route_id].cleanup()
        self.routes.clear()
        for g_id in self.granules:
            self.granules[g_id].delete()
        self.granules.clear()
        self.tl2vdt.clear()
        self.boardData.clear()

    def clear_rpc_id(self):
        self.invokeId = 0
        self.getallId = None
        self.getconfigdataId = None
        self.pingId = None
        self.subscribeId = None
        self.tracked_vehicle = None
        self.getVdtId = []
        self.getRouteGeometryId = []
        self.getDirectionsGeometryId = []

    def startConnection(self):
        debugPrint('startConnection')
        if self.socket.state() == QtNetwork.QTcpSocket.UnconnectedState:
            self.socket.connectToHost(self.connection[0], int(self.connection[1]))

    def rpcInvoke(self, method, **params):
        self.invokeId += 1
        if self.invokeId > 1024:
            self.invokeId = 1
        out_data = {"jsonrpc": "2.0", "method": method, "id": self.invokeId}
        if len(params):
            out_data["params"] = params
        out_data = json.dumps(out_data, ensure_ascii=False)
        self.socket.write(out_data + '\n')
        debugPrint('self.socket.write ' + out_data)
        return self.invokeId

    def rpcInvokeWA(self, method, **params):
        rpc_id = self.rpcInvoke(method, **params)
        self.rpcAnswers[rpc_id] = None
        timer = QtCore.QTime(0,0)
        timer.start()
        try:
            while timer.elapsed() < 7000:
                QtGui.QApplication.processEvents(QtCore.QEventLoop.WaitForMoreEvents)
                result = self.rpcAnswers.get(rpc_id, None)
                if result is not None:
                    return result
                if self.socket.bytesAvailable()>0:
                    self.socketReadyRead()
        finally:
            self.rpcAnswers.pop(rpc_id, None)

    def socketStateChanged(self, state):
        debugPrint('socketStateChanged(%d)' % state)
        if state == QtNetwork.QAbstractSocket.ConnectedState:
            self.getallId = self.rpcInvoke("getAlldata")
            self.getconfigdataId = self.rpcInvoke("getConfigData")
            self.servicePingTimer.start(PriorityClient.MAX_SERVICE_TIMEOUT)
            on_connected = getattr(self,'on_connected',None)
            if on_connected is not None:
                on_connected()
        elif state == QtNetwork.QAbstractSocket.UnconnectedState:
            self.servicePingTimer.stop()
            self.clearItems()
            self.reconnTimer.start()
            self.rcvbuf = ''
            self.rpcAnswers = {}
            self.clear_rpc_id()
            self.zoomRequest = None

    def sceneZoomChanged(self, level):
        ilevel = int(level)
        if ilevel != self.lastZoom:
            if self.zoomRequest is None:
                self.zoomRequest = ilevel
            else:
                self.zoomRequest = ilevel
                return
            self.lastZoom = ilevel
            try:
                for v_name in self.vehicles:
                    if self.zoomRequest != ilevel:
                        break
                    veh = self.vehicles[v_name]
                    if veh.iconItem:
                        veh.iconItem.lastZoom = ilevel
                        veh.iconItem.updateShape()
                    #QtCore.QCoreApplication.processEvents()
                for gr in self.granules.values():
                    if self.zoomRequest != ilevel:
                        break
                    gr.zoomChanged(self.lastZoom)
                    #QtCore.QCoreApplication.processEvents()
                if self.zoomRequest != ilevel:
                    ilevel = self.zoomRequest
                    self.zoomRequest = None
                    self.sceneZoomChanged(ilevel)
            finally:
                self.zoomRequest = None

    def servicePingTimeout(self):
        debugPrint('servicePingTimeout(), pingId=%s' % self.pingId)
        if self.pingId:
            self.pingId = None
            self.servicePingTimer.stop()
            self.socket.disconnectFromHost()
        else:
            self.pingId = self.rpcInvoke("ping")
            self.servicePingTimer.start(2000)

    def socketReadyRead(self):
        if not self.running:
            return
        data = str(self.socket.readAll().data().decode('utf8'))
        self.rcvbuf += data
#        debugPrint('socketReadyRead(). data=%s' % data.strip())
        start = 0
        while self.running:
            start = self.rcvbuf.find('{', start)
            if start < 0:
                self.rcvbuf = ''
                break
            end = self.rcvbuf.find('\n', start)
            if end < 0:
                if start > 0:
                    self.rcvbuf = self.rcvbuf[start:]
                break
            try:
                self.servicePingTimer.start(PriorityClient.MAX_SERVICE_TIMEOUT)
                j_data = json.loads(self.rcvbuf[start:end])
                result = j_data.get("result")
                error = j_data.get("error")
                req_id = j_data.get("id", -1)
                if req_id == self.getallId:  # getalldata
                    self.getallId = None
                    self.routes = {}
                    self.priorityDefaults = result.get("defaults", {})
                    j_routes = result["routes_data"]
                    for route_id in j_routes:
                        j_route = j_routes[route_id]
                        route_descr = j_route.get("description",None)
                        if self.routeNameFilter is not None:
                            if re.fullmatch(self.routeNameFilter,route_descr) is None:
                                continue
                            self.allowedTl.update(j_route.get("tls", []))
                        route_id = int(route_id)
                        new_route = Route(self.routesItem, self.layer,
                                          str(route_id),
                                          j_route.pop("description", None),
                                          j_route.pop("geometry", None),
                                          j_route.pop("tls", []))
                        lock_state = j_route.get("locked")
                        if lock_state is not None:
                            new_route.setLocked(lock_state)
                        # Создаём подветку для бортов
                        new_route.boardItems = ObjectTreeItem(new_route)
                        new_route.boardItems.setProperty("id", PriorityClient.BOARDS_BRANCH_ID)
                        # Создаём подветку для зон вызова
                        new_route.vdetectors = ObjectTreeItem(new_route)
                        new_route.vdetectors.setProperty("id", u"Зоны вызова")
                        # Список имён бортов, назначенных данному маршруту
                        new_route.assignedBoardNames = j_route.pop("boards", [])
                        for board_name in new_route.assignedBoardNames:
                            veh = self.vehicles.get(board_name)
                            if veh:
                                self.assignBoardToRoute(veh, new_route)
                        # Добавляем ветку с новыми маршрутом в ветку с маршрутами
                        RoadCentre.kernel.itemModel.addObject(new_route, self.routesItem)
                        self.routes[route_id] = new_route
                        if self.alwaysVisibleRoutes.get(route_id, False):
                            new_route.loadGeometry()
                            new_route.setAlwaysVisible(True)

                        # Добавляем данные о зонах вызова
                        for vdt_data in j_route.pop("detectors", []):
                            new_vdt = self.vdetectors.get(vdt_data['name'])
                            if new_vdt is None:
                                new_vdt = Vdetector(new_route, self.layer, vdt_data)
                                self.vdetectors[vdt_data['name']] = new_vdt
                            new_route.addDetector(new_vdt)
                            RoadCentre.kernel.itemModel.addObject(new_vdt, new_route.vdetectors)
                            if new_vdt.tlId:
                                if new_vdt.tlId not in self.tl2vdt:
                                    self.tl2vdt[new_vdt.tlId] = [new_vdt]
                                else:
                                    self.tl2vdt[new_vdt.tlId].append(new_vdt)
                        # Доп свойства
                        for k, v in j_route.items():
                            new_route.setProperty(k, v)
                        self.checkForBoards()
                    boards_data = result.get("boards_data", {})
                    if boards_data:
                        for board_name, board_data in boards_data.items():
                            if self.vehNameFilter is not None and re.fullmatch(self.vehNameFilter,board_name) is None:
                                continue
                            self.boardData[board_name] = board_data
                            if board_name not in self.vehicles:
                                v = self.addVehicle(board_name, None, None, board_data.get("last_connected_time"))
                            else:
                                v = self.vehicles[board_name]
                            v.setProperty("allow_emergency", board_data.get("allow_emergency"))
                            locked = board_data.get("locked")
                            if locked is not None:
                                v.setProperty("locked", locked)
                                if locked:
                                    v.updateIcon()
                                    v.updateToolTip()
                    # Данные о детекторах экстренного вызова
                    edetectors = result.get("edetectors", [])
                    for edt_data in edetectors:
                        cdtId = edt_data.get('cdtId')
                        tlId = edt_data.get('tl')
                        if tlId is not None and len(self.allowedTl) > 0 and tlId not in self.allowedTl:
                            continue
                        edt_name = edt_data.get('name')
                        if edt_name is None:
                            edt_name = str(cdtId)
                        new_vdt = self.vdetectors.get(edt_name)
                        if new_vdt is None:
                            new_vdt = Vdetector(new_route, self.layer, edt_data)
                            self.vdetectors[edt_name] = new_vdt
                        RoadCentre.kernel.itemModel.addObject(new_vdt, self.emergencyDetectorsItem)
                        if new_vdt.tlId:
                            if new_vdt.tlId not in self.tl2vdt:
                                self.tl2vdt[new_vdt.tlId] = [new_vdt]
                            else:
                                self.tl2vdt[new_vdt.tlId].append(new_vdt)
                    self.subscribeId = self.rpcInvoke("subscribe")
                elif req_id == self.getconfigdataId:
                    self.getconfigdataId = None
                    tmr_config = result.get("tmr")
                    if tmr_config:
                        self.snapRadius = tmr_config.get("snap_radius")
                        for v_name, veh in self.vehicles.items():
                            veh.setSnapRadius(self.snapRadius)
                elif req_id in self.getVdtId:
                    self.getVdtId.remove(req_id)
                    vdt = self.vdetectors.get(result.get('name'))
                    if vdt:
                        vdt.vdtItem.setGeometry(result.get("geometry"))
                        vdt.max_unconnected_time = result.get("max_unconnected_time")
                        vdt.max_present_time = result.get("max_present_time")
                        vdt.min_repeate_time = result.get("min_repeate_time")
                        vdt.updateDescription({}, True)
                        vdt.vdtItem.update()
                        if vdt.dontHide:
                            vdt.vdtItem.setVisible2(True,None)
                elif req_id in self.getRouteGeometryId:
                    self.getRouteGeometryId.remove(req_id)
                    route_id = result.get('id')
                    cur_route = self.routes.get(route_id)
                    if cur_route:
                        cur_route.routeItem.setGeometry(result.get("geometry"))
                        cur_route.routeItem.update()
                        if cur_route.dontHide:
                            cur_route.routeItem.setVisible(True)
                        if cur_route.dontHide:
                            for dt in cur_route.detectors:
                                dt.dontHide = True
                            cur_route.loadDetectorsGeometry()
                elif req_id in self.getDirectionsGeometryId:
                    self.getDirectionsGeometryId.remove(req_id)
                    for obj_data in result:
                        obj_id = obj_data["id"]
                        gr = self.granules.get(obj_id)
                        sg_data = obj_data["sg_data"]
                        if gr is None:
                            pos = None
                            light = RoadCentre.project.object(obj_id)
                            if light is not None:
                                pos = light.position
                            gr = Granule(self.layer, self.lastZoom, sg_data, pos)
                            self.granules[obj_id] = gr
                        else:
                            gr.changeGeometry(sg_data)
                            gr.disabled = False
                elif req_id == self.subscribeId:
                    self.subscribeId = None
                elif req_id == self.pingId:
                    self.pingId = None
                elif error:
                    msgError('JSONRPC error %d (%s): %s' % (error.get("code"), error.get("message"), error.get("data")))
                    if req_id and req_id in self.rpcAnswers:
                        self.rpcAnswers[req_id] = (True, error)
                elif req_id and req_id in self.rpcAnswers:
                    self.rpcAnswers[req_id] = (False, result)
                elif not result:
                    self.parseRPCEvent(j_data)
                elif result:
                    msgError('Unparsed answer: req_id=%s result=%s' % (req_id, result))
                start = end
                #QtCore.QCoreApplication.processEvents()
            except Exception as e:
                msgError('Parse exception: pack:%s %s' % (data[start:end], str(e)))
                if debugMode:
                    self.stop()
                    raise
                start = end
        #debugPrint('start=%d,end=%d,len=%d'%(start,end,len(self.rcvbuf)))

    def addVehicle(self, name, rpos, pos, last_connected_time=None):
        debugPrint('addVehicle(%s,%s,%s,%s)' % (name, rpos, pos, last_connected_time))
        board_data = self.boardData.get(name, {})
        v = Vehicle(self.layer, rpos, name, board_data.get("description"))
        v.setEmergencyData(board_data.get("radius", self.priorityDefaults.get("default_vehicle_radius")),
                           board_data.get("sweepAngle", self.priorityDefaults.get("default_vehicle_angle")))
        v.setShowBoardNumber(self.showBoardNumber)
        v.lastEventTime = time.time()
        v.lastConnectedTime = last_connected_time
        self.vehicles[name] = v
        v.setParent(self.unassignedBoardsItem)
        v.setSnapRadius(self.snapRadius)
        RoadCentre.kernel.itemModel.addObject(v, self.unassignedBoardsItem)
        v.changeTextShowMode(self.textShowMode)
        v.changeSectorShowMode(self.emegrencyVehSectorShowMode)
        v.updateToolTip()
        self.checkForBoards()
        v.setPriorityState(board_data.get("priority_state", False))
        # RoadCentre.kernel.itemModel.addObjects(otreeSectionName,(v,))
        return v

    def setShowBoardNumber(self, s):
        if self.showBoardNumber == s:
            return
        for v_name, veh in self.vehicles.items():
            veh.setShowBoardNumber(s)
        self.showBoardNumber = s

    def assignBoardToRoute(self, veh, new_route):
        debugPrint(u'assignBoardToRoute(%s,%s)' % (veh, new_route))
        modelItemObject = veh.property('modelItemObject')
        modelItemObjects = veh.property('modelItemObjects')
        if modelItemObjects:
            for obj in modelItemObjects:
                obj.delete()
            modelItemObject = None
            veh.setProperty('modelItemObjects', None)
            veh.setProperty('modelItemObject', None)
        if modelItemObject:
            veh.setProperty('modelItemObject', None)
            modelItemObject.delete()
        for route_id in self.routes:
            cur_route = self.routes[route_id]
            if cur_route != new_route and veh.name in cur_route.assignedBoardNames:
                cur_route.assignedBoardNames.remove(veh.name)
                break
        if new_route:
            route_boards_item = new_route.findChild('ObjectTreeItem')
            RoadCentre.kernel.itemModel.addObject(veh, self.assignedBoardsItem)
            veh.setParent(route_boards_item)
            RoadCentre.kernel.itemModel.addObject(veh, route_boards_item)
        else:
            veh.setParent(self.unassignedBoardsItem)
            RoadCentre.kernel.itemModel.addObject(veh, self.unassignedBoardsItem)
            # veh.setPriorityState(None, False)
        veh.updateToolTip()
        veh.updateIcon()
        for vdt_id in self.vdetectors:
            vdt = self.vdetectors[vdt_id]
            if vdt.activeVehicle == veh:
                vdt.changeActiveVehicle(None)

    def checkForBoards(self):
        for veh in RoadCentre.priority.unassignedBoardsItem.findChildren('ObjectTreeItem'):
            for route_id in self.routes:
                route = self.routes[route_id]
                if veh.name in route.assignedBoardNames:
                    self.assignBoardToRoute(veh, route)
                    break

    def parseRPCEvent(self, j_data):
        try:
            method = j_data.get("method")
            params = j_data.get("params")
            if method == "boardData":
                veh_name = params["name"]
                if self.vehNameFilter is not None and re.fullmatch(self.vehNameFilter,veh_name) is None:
                    return
                v = self.vehicles.get(veh_name)
                pos = params.get("pos")
                rpos = None
                offset = params.get("offset")
                offset_s = params.get("offset_s")
                if offset and offset >= 0:
                    rpos = params.get("p_pos")
                    if not rpos:
                        rpos = params.get("rpos")
                if not rpos:
                    rpos = pos
                    on_route = False
                else:
                    on_route = True
                if not v:
                    v = self.addVehicle(veh_name, rpos, pos)
                else:
                    v.offset = offset
                    v.offset_s = offset_s
                    v.move(rpos, pos, on_route)
                v.lastEventTime = time.time()
                # speed=params.get("speed")
                course = params.get("course")
                if course is not None:
                    v.rotate(course)
                else:
                    v.iconItem.updateShape()
                priority_state = params.get("priority_state", False)
                if priority_state == 2:
                    priority_state = False
#                debugPrint("kalt=%s, priority_state=%s" % (params.get("kalt"), priority_state))
                v.setPriorityState(priority_state)
                speed = params.get("speed")
                if speed is not None:
                    v.setSpeed(speed)
                routeItem = v.parent().parent()
                if routeItem is not None and hasattr(routeItem, "id"):
                    invalid_widgets = []
                    for w in self.timelineWidgets:
                        try:
                            tm_string = params.get("time")
                            if tm_string is not None and tm_string[-1] != "Z" and tm_string[-1] != "z":
                                tm_string += "Z"
                            w.putBoardData(veh_name, routeItem.id,
                                           QtCore.QDateTime.fromString(
                                               tm_string, QtCore.Qt.ISODate).toLocalTime(), offset_s, priority_state)
                        except:
                            invalid_widgets.append(w)
                    for w in invalid_widgets:
                        self.removeTimelineWidget(w)
                        # if v.captionItem:
                        #    v.captionItem.setPlainText(str(params.get("kalt",-1)))
            elif method == "vdtStateChanged":
                vdt = self.vdetectors.get(params.get('name'))
                debugPrint("vdtStateChanged(vdt=%s): %s" % (vdt, params))
                if self.on_vdt_event:
                    self.on_vdt_event(params)
                if vdt:
                    # На одном СО приоритет может иметь только одно ТС, поэтому
                    # на всех зонах вызова кроме текущей убираем активные борты
                    if vdt.tlId:
                        vdt_list = self.tl2vdt.get(vdt.tlId, [])
                        for cur_vdt in vdt_list:
                            if cur_vdt != vdt:
                                cur_vdt.changeActiveVehicle(None)
                    board_name = params.get('board')
                    if board_name:
                        v = self.vehicles.get(board_name)
                        if v:
                            # v.setPriorityState(vdt, True)
                            vdt.changeActiveVehicle(v)
                    vdt.changeState(params.get('state'), params.get('locked'))
            elif method == "boardAssigned":
                veh = self.vehicles.get(params["name"])
                route = self.routes.get(params["new_route_id"])
                if veh:
                    if route and veh.name not in route.assignedBoardNames:
                        route.assignedBoardNames.append(veh.name)
                    self.assignBoardToRoute(veh, route)
                else:
                    debugPrint("vehicle not found in boardAssigned(route=%s,veh=%s): %s" % (route, veh, params))
            elif method == "boardConnected":
                veh_name = params["name"]
                if self.vehNameFilter is not None and re.fullmatch(self.vehNameFilter,veh_name) is None:
                    return
                v = self.vehicles.get(veh_name)
                if not v:
                    self.addVehicle(veh_name, None, None)
                else:
                    last_pos = (v.iconItem.pos.x(), v.iconItem.pos.y())
                    v.move(last_pos, last_pos, v.onRoute)
            elif method == "boardDisconnected":
                veh_name = params["name"]
                v = self.vehicles.get(veh_name)
                if v:
                    v.cleanup()
            elif method == "boardPriorityStateChanged":
                veh_name = params["name"]
                v = self.vehicles.get(veh_name)
                if v:
                    prio_state = params.get("state", False)
                    if prio_state == 2:
                        prio_state = False
                    v.setPriorityState(prio_state)
            elif method == "routePropsChanged":
                route_obj = self.routes.get(params["id"])
                if route_obj:
                    for k, v in params.get("props", {}).items():
                        if k == "locked":
                            route_obj.setLocked(v)
                            continue
                        route_obj.setProperty(k, v)
                        if k == "description":
                            route_obj.updateType()
                    RoadCentre.gui.treeView.model().layoutChanged.emit()
            elif method == "boardPropsChanged":
                board_name = params["name"]
                board_data = self.boardData.get(board_name)
                if not board_data:
                    board_data = {}
                    self.boardData[board_name] = board_data
                veh = self.vehicles.get(board_name)
                need_update_shape = False
                for k, v in params.get("props", {}).items():
                    if not need_update_shape and k in ("sweepAngle", "radius"):
                        need_update_shape = True
                    board_data[k] = v
                    if veh:
                        veh.setProperty(k, v)
                veh.updateToolTip()
                veh.updateIcon()
                if need_update_shape:
                    veh.setEmergencyData(board_data.get("radius", self.priorityDefaults.get("default_vehicle_radius")),
                                         board_data.get("sweepAngle",
                                                        self.priorityDefaults.get("default_vehicle_angle")))
                    veh.iconItem.updateShape()
                RoadCentre.gui.treeView.model().layoutChanged.emit()
            elif method == "dtConfigEvent":
                vdt = self.vdetectors.get(params["name"])
                if vdt:
                    vdt.updateDescription(params, update=True)
                else:
                    for vdt in self.vdetectors.values():
                        if vdt.vdtItem.is_edetector:
                            if vdt.cdtId == params.get("cdtId"):
                                vdt.updateDescription(params, update=True)
                        elif vdt.enterId == params.get("enter") and vdt.leaveId == params.get("leave"):
                            vdt.updateDescription(params, update=True)
                            break
            else:
                debugPrint("unknown event:" + str(method))
        except Exception as e:
            if debugMode:
                raise
            msgError('Parse event exception: pack:%s %s' % (j_data, str(e)))
            return

    def setDtLocked(self, dt, locked):
        vdt = self.vdetectors.get(dt)
        if vdt is None and type(dt) == int:
            vdt = self.vdetectors.get(str(dt))
        if vdt:
            self.rpcInvoke("controlVdt", name=dt, lock=locked)

    def setDtActive(self, dt, active):
        vdt = self.vdetectors.get(dt)
        if vdt is None and type(dt) == int:
            vdt = self.vdetectors.get(str(dt))
        if vdt:
            self.rpcInvoke("activateVdt", name=dt, activate=active)

    def showSnapRadius(self, s):
        for v_name, veh in self.vehicles.items():
            veh.setSnapRadiusVisible(s)

    def objectNotification(self, notify, light):
        if notify.id != 5:
            return
        gr = self.granules.get(light.id)
        if gr is None:
            return
        sg_state = notify.data[1]
        if sg_state == 1 or sg_state == 2 or sg_state == 3:
            sg_state = Granule.GREEN
        else:
            sg_state = Granule.RED
        gr.setSgState(notify.data[0], sg_state)

    def objectConnected(self, conn, light):
        if not conn:
            gr = self.granules.get(light.id)
            if gr is not None:
                gr.resetStates()

    def itemDropComplete(self, data, parent):
        debugPrint('itemDropComplete(data=%s,parent=%s)' % (data, parent))
        stream = QtCore.QDataStream(data, QtCore.QIODevice.ReadOnly)
        board_list = []
        while not stream.atEnd():
            board_list.append(stream.readQString())
        board_count = len(board_list)
        if not board_count:
            return
        if board_count == 1:
            board_str = u"борт %s" % board_list[0]
        else:
            board_str = u"борты %s" % ','.join(board_list)
        route_id = 0
        if parent.data() != self.unassignedBoardsItem.id:
            route_id = int(parent.data())
            query_str = u"Присоединить %s к маршруту %s" % (board_str, route_id)
        else:
            query_str = u"Отсоединить %s от маршрута" % board_str
        res = QtGui.QMessageBox.question(0, u"Вопрос", query_str, QtGui.QMessageBox.Yes | QtGui.QMessageBox.No)
        if res == QtGui.QMessageBox.Yes:
            for board_name in board_list:
                if route_id:
                    assignBoard(board_name, route_id)
                else:
                    self.unassignBoard(board_name)

    def unassignBoard(self, board_name):
        self.rpcInvoke("unassignBoard", board=board_name)

    def assignBoard(self, board_name, route_id):
        self.rpcInvoke("assignBoardToRoute", board=board_name, route=route_id)

    def lockBoard(self, board_name):
        self.rpcInvoke("changeBoardProperties", board_name=board_name, props={"locked": True})

    def unlockBoard(self, board_name):
        self.rpcInvoke("changeBoardProperties", board_name=board_name, props={"locked": False})

    def lockRoute(self, route_id):
        self.rpcInvoke("changeRouteProperties", route_id=int(route_id), props={"locked": True})

    def unlockRoute(self, route_id):
        self.rpcInvoke("changeRouteProperties", route_id=int(route_id), props={"locked": False})

    def modelDataChanged(self, index, value, role):
        debugPrint('modelDataChanged(index=%s,value=%s,role=%d)' % (index, value, role))
        parent = index.parent()
        if not parent:
            return
        parent_id = parent.data()
        if parent_id == self.routesItem.id:
            route_id = int(index.sibling(index.row(), 0).data(0))
            self.rpcInvoke("changeRouteProperties", route_id=route_id, props={"description": value.encode('utf8')})
        if parent_id == self.unassignedBoardsItem.id or parent_id == self.assignedBoardsItem.id or parent_id == PriorityClient.BOARDS_BRANCH_ID:
            self.rpcInvoke("changeBoardProperties", board_name=index.sibling(index.row(), 0).data(0),
                           props={"description": value})

    def changeTextShowMode(self, mode):
        if self.textShowMode == mode:
            return
        self.textShowMode = mode
        for v_name in self.vehicles:
            self.vehicles[v_name].changeTextShowMode(mode)

    def changeEVSectorShowMode(self, mode):
        if self.emegrencyVehSectorShowMode == mode:
            return
        self.emegrencyVehSectorShowMode = mode
        for v_name in self.vehicles:
            self.vehicles[v_name].changeSectorShowMode(mode)

    def changeZoomValues(self, max_zoom, min_zoom, poi_scale):
        update_flag = False
        if VehicleItem.MAX_ZOOM != max_zoom:
            VehicleItem.MAX_ZOOM = max_zoom
            update_flag = True
        if VehicleItem.MIN_ZOOM != min_zoom:
            VehicleItem.MIN_ZOOM = min_zoom
            update_flag = True
        if VehicleItem.POI_SCALE != poi_scale:
            VehicleItem.POI_SCALE = poi_scale
            update_flag = True
        if not update_flag:
            return
        for v_name in self.vehicles:
            veh = self.vehicles[v_name]
            if veh.iconItem:
                veh.iconItem.updateShape()

    def editVdt(self, vdt):
        err_title = u"Ошибка"
        if vdt.vdtItem.is_edetector:
            vdt_data = self.rpcInvokeWA("getVdt", name=vdt.id)
        else:
            vdt_data = self.rpcInvokeWA("getVdt", enter=vdt.enterId, leave=vdt.leaveId)
        if not vdt_data or vdt_data[0]:
            err_str = u"Истекло время ожидания ответа"
            if vdt_data and vdt_data[0]:
                err_str = u"Ошибка удалённого вызова (%s):%s" % (vdt_data[1].get("code"), vdt_data[1].get("data"))
            QtGui.QMessageBox.critical(0, err_title, err_str)
            return
        vdt_data = vdt_data[1]
        dlg = widgetFromUi('vdtprops.ui')
        name = dlg.findChild('QLineEdit', 'name')
        max_present_time = dlg.findChild('QSpinBox', 'max_present_time')
        max_unconnected_time = dlg.findChild('QSpinBox', 'max_unconnected_time')
        min_repeate_time = dlg.findChild('QSpinBox', 'min_repeate_time')
        activations_before_change = dlg.findChild('QSpinBox', 'activations_before_change')
        tl = dlg.findChild('QSpinBox', 'tl')
        call_by_button = dlg.findChild('QCheckBox', 'call_by_button')
        prog = dlg.findChild('QSpinBox', 'prog')
        phase = dlg.findChild('QSpinBox', 'phase')
        crossWI = dlg.findChild('QSpinBox', 'crossWI')
        crossWO = dlg.findChild('QSpinBox', 'crossWO')
        no_control = dlg.findChild('QRadioButton', 'no_control')
        cond_table = dlg.findChild('QTableWidget', 'cond_table')

        def checkTlId():
            valid = no_control.isChecked() or tl.value != 0
            pal = QtGui.QSpinBox().palette
            if not valid:
                pal.setColor(QtGui.QPalette.Base, QtCore.Qt.red)
            tl.setPalette(pal)

        tl.connect("valueChanged(QString)", checkTlId)
        no_control.connect("toggled(bool)", checkTlId)

        intattr = lambda v: v if v else 0
        intattr2 = lambda v: v if v else -1
        name.text = vdt_data.get("name", "")
        max_present_time.value = intattr(vdt_data.get("max_present_time", 0))
        max_unconnected_time.value = intattr(vdt_data.get("max_unconnected_time", 0))
        min_repeate_time.value = intattr(vdt_data.get("min_repeate_time", 0))
        activations_before_change.value = intattr(vdt_data.get("activations_before_change", 1))
        cond_values = vdt_data.get("buttonStateData", {})
        tl.value = intattr(vdt_data.get("tl", 0))
        call_by_button.checked = vdt_data.get("priorityByButton", False)
        prog.value = intattr(vdt_data.get("prog", 0))
        phase.value = intattr(vdt_data.get("phase", 0))
        crossWI.value = intattr2(vdt_data.get("crossWI", -1))
        crossWO.value = intattr2(vdt_data.get("crossWO", -1))
        if call_by_button.checked and len(cond_values) > 0:
            dlg.findChild('QRadioButton', 'by_cond').checked = True
        elif prog.value:
            dlg.findChild('QRadioButton', 'prog_btn').checked = True
        elif phase.value:
            dlg.findChild('QRadioButton', 'phase_btn').checked = True
        row = 0
        for phase_id in cond_values:
            cond_table.setItem(row, 0, QtGui.QTableWidgetItem(str(phase_id)))
            prog_id = cond_values[phase_id]
            if prog_id is not None:
                cond_table.setItem(row, 1, QtGui.QTableWidgetItem(str(prog_id)))
            row += 1
        if dlg.exec_() != QtGui.QDialog.Accepted:
            return
        changed_props = {}
        if max_present_time.value != vdt_data.get("max_present_time"):
            changed_props["max_present_time"] = max_present_time.value
        if max_unconnected_time.value != vdt_data.get("max_unconnected_time"):
            changed_props["max_unconnected_time"] = max_unconnected_time.value
        if min_repeate_time.value != vdt_data.get("min_repeate_time"):
            changed_props["min_repeate_time"] = min_repeate_time.value
        if activations_before_change.value != vdt_data.get("activations_before_change"):
            changed_props["activations_before_change"] = activations_before_change.value
        if tl.value != vdt_data.get("tl"):
            changed_props["tl"] = tl.value
        if call_by_button.checked != vdt_data.get("priorityByButton"):
            changed_props["priorityByButton"] = call_by_button.checked
        if crossWI.value != vdt_data.get("crossWI"):
            changed_props["crossWI"] = crossWI.value if crossWI.value >= 0 else None
        if crossWO.value != vdt_data.get("crossWO"):
            changed_props["crossWO"] = crossWO.value if crossWO.value >= 0 else None
        new_cond_values = {}
        if cond_table.enabled:
            for row in range(cond_table.rowCount):
                item = cond_table.item(row, 0)
                if item is None:
                    continue
                ph_id = item.text()
                if not ph_id.isdigit():
                    continue
                item = cond_table.item(row, 1)
                pr_id = None
                if item is not None:
                    pr_id = item.text()
                    if not pr_id.isdigit():
                        pr_id = None
                    else:
                        pr_id = int(pr_id)
                new_cond_values[int(ph_id)] = pr_id

        if new_cond_values != cond_values:
            changed_props["buttonStateData"] = new_cond_values
        prog_val = prog.value if prog.enabled else None
        if prog_val != vdt_data.get("prog"):
            changed_props["prog"] = prog_val
        phase_val = phase.value if phase.enabled else None
        if phase_val != vdt_data.get("phase"):
            changed_props["phase"] = phase_val
        if prog_val == 0:
            QtGui.QMessageBox.critical(0, err_title, u"Необходимо указать номер программы")
            return
        if phase_val == 0:
            QtGui.QMessageBox.critical(0, err_title, u"Необходимо указать номер фазы")
            return
        if name.text != vdt_data.get("name"):
            changed_props["name"] = name.text
        if not len(changed_props):
            return
        if vdt.vdtItem.is_edetector:
            changed_props["cdtId"] = vdt.cdtId
        else:
            changed_props["enter"] = vdt.enterId
            changed_props["leave"] = vdt.leaveId
        self.rpcInvoke("changeVdt", **changed_props)

    def editVehicles(self, vehicles):
        veh_count = len(vehicles)
        if veh_count == 0:
            return
        veh = vehicles[0]
        dlg = widgetFromUi('boardprops.ui')
        if veh_count == 1:
            dlg.windowTitle = u'Свойства борта ' + veh.id
        else:
            dlg.windowTitle = u'Свойства нескольких бортов'
        description = dlg.findChild('QLineEdit', 'description')
        locked = dlg.findChild('QCheckBox', 'locked')
        allow_emergency = dlg.findChild('QCheckBox', 'allow_emergency')
        radius = dlg.findChild('QSpinBox', 'radius')
        sweep_angle = dlg.findChild('QSpinBox', 'sweep_angle')

        if veh.property("description") is not None:
            description.text = veh.property("description")
        if veh.property("locked"):
            locked.checked = True
        if veh.property("allow_emergency"):
            allow_emergency.checked = True
        radius.value = veh.iconItem.viewRadius
        sweep_angle.value = veh.iconItem.viewAngle
        if dlg.exec_() != QtGui.QDialog.Accepted:
            return
        for cur_veh in vehicles:
            changed_props = {}
            if description.text != cur_veh.property("description"):
                changed_props["description"] = description.text
            if bool(cur_veh.property("locked")) != locked.checked:
                changed_props["locked"] = locked.checked
            if bool(cur_veh.property("allow_emergency")) != allow_emergency.checked:
                changed_props["allow_emergency"] = allow_emergency.checked
            if radius.value != cur_veh.iconItem.viewRadius:
                changed_props["radius"] = radius.value
            if sweep_angle.value != cur_veh.iconItem.viewAngle:
                changed_props["sweepAngle"] = sweep_angle.value
            self.rpcInvoke("changeBoardProperties", board_name=cur_veh.id, props=changed_props)

    def openTrackUrl(self, board_name, date_str):
        err_title = u"Ошибка"
        url_data = self.rpcInvokeWA("getBoardTrackUrl", name=board_name, date=date_str)
        if not url_data or url_data[0]:
            err_str = u"Истекло время ожидания ответа"
            if url_data and url_data[0]:
                err_str = u"Ошибка удалённого вызова (%s):%s" % (url_data[1].get("code"), url_data[1].get("data"))
            QtGui.QMessageBox.critical(0, err_title, err_str)
            return
        url = url_data[1]
        host_start = url.find('//')
        if host_start >= 0:
            host_start += 2
            host_end = url.find(':', host_start)
            if host_end:
                url = url[:host_start] + self.connection[0] + url[host_end:]
        thread = Thread(target=webbrowser.open, args=(url,))
        thread.start()
        thread.join(0.3)

    def showTimeline(self, r_id, archive=False):
        if not findSynchronizerForTimeline:
            return
        err_title = u"Ошибка"
        timeline_data = self.rpcInvokeWA("getTimelineData2", route=int(r_id))
        if timeline_data is None or timeline_data[0]:
            err_str = u"Истекло время ожидания ответа"
            if timeline_data is not None and timeline_data[0]:
                err_str = u"Ошибка удалённого вызова (%s):%s" % (
                    timeline_data[1].get("code"), timeline_data[1].get("data"))
            QtGui.QMessageBox.critical(0, err_title, err_str)
            return
        doc_window = RoadCentre.plugins.TimelinePlugin.createTimelineWidget(mainWindow, int(r_id), timeline_data[1],
                                                                            archive)
        tlw = doc_window.findChild('TimelineWidget')
        self.addTimelineWidget(tlw)
        tlw.connect('destroyed(QObject *)', self.removeTimelineWidget)
        findSynchronizerForTimeline(tlw)
        tl_list = []
        for oid in tlw.getRoutesTlId():
            light = RoadCentre.project.object(oid)
            if light is not None:
                tl_list.append(light)
        tlw.setRoutesTl(tl_list)
        docTimelineWidget(doc_window)

    def getVdtGeometry(self, enterId, leaveId):
        self.getVdtId.append(self.rpcInvoke("getVdt", enter=enterId, leave=leaveId, with_geometry=True))

    def getRouteGeometry(self, route_id):
        self.getRouteGeometryId.append(self.rpcInvoke("getRouteGeometry", id=int(route_id)))

    def getDirectionsGeometry(self, obj_id_list):
        self.getDirectionsGeometryId.append(self.rpcInvoke("getDirectionsGeometry", obj_id=obj_id_list))

    def resetDirectionsForObject(self, obj_id, disable_granule=False):
        gr = self.granules.get(obj_id)
        if gr:
            gr.resetStates(disable_granule)

    def set_vehicle_tracking(self,veh,track):
        if veh != self.tracked_vehicle:
            if self.tracked_vehicle:
                self.tracked_vehicle.set_vehicle_tracking(False)
            self.tracked_vehicle = veh
        veh.set_vehicle_tracking(track)
        if not track:
            self.tracked_vehicle = None


def editSettings():
    dlg = widgetFromUi('prioritysetup.ui')
    enable_priority = dlg.findChild('QCheckBox', 'enable_priority')
    host = dlg.findChild('QLineEdit', 'host')
    port = dlg.findChild('QSpinBox', 'port')
    max_zoom = dlg.findChild('QSpinBox', 'max_zoom')
    min_zoom = dlg.findChild('QSpinBox', 'min_zoom')
    pict_scale = dlg.findChild('QDoubleSpinBox', 'pict_scale')
    no_show_text = dlg.findChild('QRadioButton', 'no_show_text')
    show_text_on_sel = dlg.findChild('QRadioButton', 'show_text_on_sel')
    show_text = dlg.findChild('QRadioButton', 'show_text')
    no_show_ems = dlg.findChild('QRadioButton', 'no_show_ems')
    show_ems_on_sel = dlg.findChild('QRadioButton', 'show_ems_on_sel')
    show_ems = dlg.findChild('QRadioButton', 'show_ems')
    VehNameFilter = dlg.findChild('QLineEdit', 'VehNameFilter')
    RouteNameFilter = dlg.findChild('QLineEdit', 'RouteNameFilter')
    showBoardNumber = dlg.findChild('QCheckBox', 'showBoardNumber')

    enable_priority.checked = int(userSettings.value('Priority/Enabled', 0)) == 1
    showBoardNumber.checked = int(userSettings.value('Priority/ShowBoardNumber', 0)) == 1

    prio_settings = RoadCentre.project.settings
    if prio_settings is None:
        prio_settings = userSettings
    control_connection = prio_settings.value('Priority/ControlServer')
    if control_connection:
        try:
            host.text, port_str = control_connection.split(':')
            port.value = int(port_str)
        except:
            pass
    max_zoom.value = int(userSettings.value("Priority/max_zoom", VehicleItem.MAX_ZOOM))
    min_zoom.value = int(userSettings.value("Priority/min_zoom", VehicleItem.MIN_ZOOM))
    pict_scale.value = float(userSettings.value("Priority/pict_scale", VehicleItem.POI_SCALE))
    text_show_value = int(userSettings.value("Priority/text_show_value", Vehicle.TEXT_SHOW_SELECTED))
    ems_show_value = int(userSettings.value("Priority/ems_show_value", Vehicle.TEXT_SHOW_SELECTED))
    VehNameFilter.text = userSettings.value('Priority/VehNameFilter', '')
    RouteNameFilter.text = userSettings.value('Priority/RouteNameFilter', '')
    if text_show_value == Vehicle.TEXT_SHOW_SELECTED:
        show_text_on_sel.checked = True
    elif text_show_value == Vehicle.TEXT_SHOW_ALWAYS:
        show_text.checked = True
    else:
        no_show_text.checked = True
    if ems_show_value == Vehicle.TEXT_SHOW_SELECTED:
        show_ems_on_sel.checked = True
    elif ems_show_value == Vehicle.TEXT_SHOW_ALWAYS:
        show_ems.checked = True
    else:
        no_show_ems.checked = True
    if dlg.exec_() != QtGui.QDialog.Accepted:
        return
    userSettings.setValue("Priority/Enabled", 1 if enable_priority.checked else 0)
    userSettings.setValue("Priority/ShowBoardNumber", 1 if showBoardNumber.checked else 0)
    userSettings.setValue("Priority/max_zoom", max_zoom.value)
    userSettings.setValue("Priority/min_zoom", min_zoom.value)
    userSettings.setValue("Priority/pict_scale", pict_scale.value)
    veh_filter_changed = userSettings.value('Priority/VehNameFilter', '') != VehNameFilter.text
    if veh_filter_changed:
        userSettings.setValue("Priority/VehNameFilter", VehNameFilter.text)
    route_filter_changed = userSettings.value('Priority/RouteNameFilter', '') != RouteNameFilter.text
    if route_filter_changed:
        userSettings.setValue("Priority/RouteNameFilter", RouteNameFilter.text)

    text_show_value = Vehicle.TEXT_SHOW_NONE
    if show_text_on_sel.checked:
        text_show_value = Vehicle.TEXT_SHOW_SELECTED
    elif show_text.checked:
        text_show_value = Vehicle.TEXT_SHOW_ALWAYS
    userSettings.setValue("Priority/text_show_value", text_show_value)

    ems_show_value = Vehicle.TEXT_SHOW_NONE
    if show_ems_on_sel.checked:
        ems_show_value = Vehicle.TEXT_SHOW_SELECTED
    elif show_ems.checked:
        ems_show_value = Vehicle.TEXT_SHOW_ALWAYS
    userSettings.setValue("Priority/ems_show_value", ems_show_value)
    str_port = str(port.value)
    prio_settings = RoadCentre.project.settings
    if prio_settings is None:
        prio_settings = userSettings
    prio_settings.setValue('Priority/ControlServer', ':'.join([host.text, str_port]))

    if not hasattr(RoadCentre, 'priority'):
        RoadCentre.priority=PriorityClient((host.text,str_port))
    client = getattr(RoadCentre, 'priority', None)
    if client:
        client.enabled = enable_priority.checked
        client.setShowBoardNumber(showBoardNumber.checked)
        if not client.enabled and client.running:
            client.stop()
        client.changeTextShowMode(text_show_value)
        client.changeEVSectorShowMode(ems_show_value)
        if client.connection[0] != host.text or client.connection[1] != str_port:
            client.stop()
            client.connection[0] = host.text
            client.connection[1] = str_port
            client.start()
        elif veh_filter_changed or route_filter_changed:
            client.stop()
        if client.enabled and project.connection.state == project.connection.ReadyState:
            client.start()
        client.changeZoomValues(max_zoom.value, min_zoom.value, pict_scale.value)
