# -*- coding: utf-8 -*-
from priority.geometry import xyToGeo, geoToXY, MeterToDistance
from PythonQt import QtCore, QtGui
from PythonQt.Road import ObjectTreeItem
import RoadCentre
from menu import SubMenu
from gui import osmView


class VdetectorItem(QtGui.QGraphicsItem):
    def __init__(self, layer, args):
        QtGui.QGraphicsItem.__init__(self)
        self.setFlags(QtGui.QGraphicsItem.ItemIsSelectable)
        self.setCursor(QtCore.Qt.ArrowCursor)
        self.enter = QtGui.QPainterPath()
        self.leave = QtGui.QPainterPath()
        self.connector = QtGui.QPainterPath()
        self.shapePath = QtGui.QPainterPath()
        self.shapePath.setFillRule(QtCore.Qt.WindingFill)
        self.bound = QtCore.QRectF()
        self.arrow = QtGui.QPolygonF()
        self.setParentItem(layer)
        self.state = args.get('state', False)
        self.locked = args.get('locked', False)
        self.setVisible(False)
        self.setZValue(0.1)
        self.has_geometry = False
        self.is_edetector = False
        self.vis_by_route = None

        self.setGeometry(args.get('geometry'))
        if not self.has_geometry:
            self.setEdtGeometry(args.get("pos"), args.get("radius"), args.get("directionAngle"), args.get("sweepAngle"))

    def setGeometry(self, geom):
        if geom is None:
            return
        self.prepareGeometryChange()
        p1 = QtCore.QPointF(geom[0][0], geom[0][1])
        p12 = QtCore.QPointF(geom[-2][0], geom[-2][1])
        p2 = QtCore.QPointF(geom[-1][0], geom[-1][1])
        ellipse_size = 3.0
        arrow_height = MeterToDistance(ellipse_size / 2.0 + 1, p12) / 2.0
        arrow_len = arrow_height * 4.0

        fp1 = geoToXY(p1)
        fp12 = geoToXY(p12)
        fp2 = geoToXY(p2)
        last_line = QtCore.QLineF(fp12, fp2)
        last_line.setLength(last_line.length() - arrow_len)
        p2 = xyToGeo(last_line.p2())
        geom[-1][0] = p2.x()
        geom[-1][1] = p2.y()
        skel = QtGui.QPainterPath()
        pfp = None
        for pp in geom:
            fp = geoToXY(QtCore.QPointF(pp[0], pp[1]))
            if skel.elementCount() == 0:
                skel.moveTo(fp)
            else:
                if QtCore.QLineF(pfp, fp).length() < 0.001:
                    continue
                skel.lineTo(fp)
            pfp = fp
        if not skel.isEmpty():
            skel.setElementPositionAt(skel.elementCount()-1, fp.x(), fp.y())
        stroker = QtGui.QPainterPathStroker()
        stroker.setWidth(MeterToDistance(ellipse_size / 2.0, p1))
        # полигон получится с изломами, но simplified вызывать нельзя, т.к. при некоторых сочетаниях координат
        # и ширины обводки упрощение выдаёт нулевой путь
        poly = stroker.createStroke(skel)
        self.connector = QtGui.QPainterPath()
        self.connector.setFillRule(QtCore.Qt.WindingFill)
        for i in range(poly.elementCount()):
            pt = xyToGeo(poly.elementAt(i).operator_cast_QPointF())
            if i:
                self.connector.lineTo(pt)
            else:
                self.connector.moveTo(pt)
        el_size = MeterToDistance(ellipse_size, p1)
        rc = QtCore.QRectF(fp1.x() - el_size / 2.0, fp1.y() - el_size / 2.0, el_size, el_size)
        rc.setTopLeft(xyToGeo(rc.topLeft()))
        rc.setBottomRight(xyToGeo(rc.bottomRight()))
        self.enter.addEllipse(rc)

        el_size = MeterToDistance(ellipse_size, p2)
        rc = QtCore.QRectF(fp2.x() - el_size / 2.0, fp2.y() - el_size / 2.0, el_size, el_size)
        rc.setTopLeft(xyToGeo(rc.topLeft()))
        rc.setBottomRight(xyToGeo(rc.bottomRight()))
        self.leave.addEllipse(rc)

        direction = QtCore.QLineF(fp12, fp2)
        t = QtGui.QTransform()
        t.translate(direction.p2().x(), direction.p2().y())
        t.rotate(-direction.angle())
        self.arrow.append(xyToGeo(t.map(QtCore.QPointF(-arrow_len, -arrow_height))))
        self.arrow.append(xyToGeo(direction.p2()))
        self.arrow.append(xyToGeo(t.map(QtCore.QPointF(-arrow_len, arrow_height))))

        self.shapePath.addPath(self.enter)
        self.shapePath.addPath(self.leave)
        self.shapePath.addPath(self.connector)
        self.shapePath.addPolygon(self.arrow)
        self.bound = self.shapePath.boundingRect()
        self.has_geometry = True

    def setEdtGeometry(self, pos, radius, direction, sweep):
        if pos is None or radius is None or direction is None or sweep is None:
            return
        pos = QtCore.QPointF(*pos)
        self.prepareGeometryChange()
        self.shapePath = QtGui.QPainterPath()
        wh = MeterToDistance(radius, pos) * 2.0
        circle_diam = MeterToDistance(2, pos)
        sector_rect = QtCore.QRectF(0, 0, wh, wh)
        sector_rect.moveCenter(geoToXY(pos))
        self.shapePath.addEllipse(sector_rect.center(), circle_diam, circle_diam)
        self.shapePath.moveTo(sector_rect.center())
        self.shapePath.arcTo(sector_rect, - direction + sweep / 2.0, - sweep)
        self.shapePath.closeSubpath()
        for i in range(self.shapePath.elementCount()):
            geo_pos = xyToGeo(self.shapePath.elementAt(i).operator_cast_QPointF())
            self.shapePath.setElementPositionAt(i, geo_pos.x(), geo_pos.y())
        self.setPos(pos)
        self.shapePath.translate(QtCore.QPointF() - pos)
        self.bound = self.shapePath.boundingRect()
        self.is_edetector = True
        self.has_geometry = True

    def boundingRect(self):
        return self.bound

    def shape(self):
        return self.shapePath

    def paint(self, painter, opt, w):
        pen = painter.pen()
        if not pen.isCosmetic():
            pen.setCosmetic(True)
        cur_color = QtCore.Qt.lightGray
        if self.state == True:
            cur_color = QtCore.Qt.green
        elif self.state == False:
            cur_color = QtCore.Qt.darkGray
        if self.locked:
            cur_color = QtCore.Qt.red
        painter.setBrush(QtGui.QBrush(cur_color))
        if self.isSelected():
            pen.setStyle(QtCore.Qt.SolidLine)
            pen.setWidth(2)
            pen.setCosmetic(True)
        if self.is_edetector:
            painter.setOpacity(0.6)
            painter.drawPath(self.shapePath)
        else:
            # двойная прорисовка нужна, чтобы скрыть самопересечения на изгибах.
            # По идее simplified должна решать проблему, но в некоторых случаях она выдаёт нулевой путь
            painter.drawPath(self.connector)
            painter.fillPath(self.connector, cur_color)
            painter.drawPath(self.enter)
            painter.drawPath(self.leave)
            painter.drawPolygon(self.arrow)

    def setVisible2(self,v,route):
        if v:
            self.vis_by_route = route
        elif route!=self.vis_by_route:
            return
        self.setVisible(v)


class Vdetector(ObjectTreeItem):
    def __init__(self, parent, layer, args):
        ObjectTreeItem.__init__(self, parent)
        self.vdtItem = VdetectorItem(layer, args)
        if self.vdtItem.is_edetector:
            self.setProperty('type', u'Детектор')
        else:
            self.setProperty('type', u'Зона вызова')
        self.vdtItem.setData(0, self)
        self.connect('selectedChanged(bool)', self.selectedChanged)
        self.setProperty("itemFlags", QtCore.Qt.ItemIsSelectable | QtCore.Qt.ItemIsEnabled)
        self.enterId = args.get('enter')
        self.leaveId = args.get('leave')
        self.tlId = args.get('tl')
        self.cdtId = args.get('cdtId')
        self.prog = None
        self.phase = None
        self.routes = []
        self.max_unconnected_time = None
        self.max_present_time = None
        self.min_repeate_time = None
        self.updateDescription(args)
        self.activeVehicle = None
        self.dontHide = False
        self.updateIcon()

    def cleanup(self):
        self.disconnect('selectedChanged(bool)', self.selectedChanged)
        for r in self.routes:
            r.removeDetector(self)
        self.routes = []
        self.vdtItem.delete()
        self.deleteLater()

    def addRoute(self,r):
        if r not in self.routes:
            self.routes.append(r)
        self.updateDescription({"name": self.id}, True)

    def updateZValue(self):
        zv = 0.1
        if self.vdtItem.state or self.vdtItem.isSelected():
            zv = 0.2
        self.vdtItem.setZValue(zv)

    def selectedChanged(self, s):
        if not s and (self.dontHide or self.parent().routeItem.isSelected()):
            return
        if s and not self.vdtItem.has_geometry:
            RoadCentre.priority.getVdtGeometry(self.enterId, self.leaveId)
        if self.vdtItem.is_edetector:
            if not self.vdtItem.state:
                self.vdtItem.setVisible2(s,None)
        else:
            self.vdtItem.setVisible2(s,None)
        self.vdtItem.setSelected(s)
        self.updateZValue()

    def changeState(self, state, locked):
        need_update = False
        if self.vdtItem.state != state:
            self.vdtItem.state = state
            self.updateZValue()
            need_update = True
        if self.vdtItem.locked != locked:
            self.vdtItem.locked = locked
            need_update = True
        if not self.vdtItem.state and self.activeVehicle:
            self.changeActiveVehicle(None)
        if self.vdtItem.is_edetector:
            self.vdtItem.setVisible2(state,None)
        if need_update:
            self.vdtItem.update()
            self.updateIcon()

    def changeActiveVehicle(self, veh):
        if self.activeVehicle == veh:
            return
        self.activeVehicle = veh

    def updateDescription(self, args, update=False):
        name = args.get('name', self.property("id") if update else None)
        if name is None:
            name = args.get('cdtId')
        self.setProperty('id', name)
        self.tlId = args.get('tl', self.tlId)
        desc_str = ""
        if self.tlId:
            desc_str = u"СО %s" % self.tlId
        prog = args.get("prog")
        if prog is None and update and "prog" not in args:
            prog = self.prog
        if prog is not None:
            if desc_str: desc_str += u", "
            desc_str += u"программа %s" % prog
        self.prog = prog
        phase = args.get("phase")
        if phase is None and update and "phase" not in args:
            phase = self.phase
        if phase is not None and prog is None:
            if desc_str: desc_str += u", "
            desc_str += u"фаза %s" % phase
        self.phase = phase
        if self.vdtItem.is_edetector:
            tooltip = u'Детектор %s' % self.id
        else:
            tooltip = u'Зона вызова %s' % self.id

        self.setProperty('description', desc_str)

        self.max_present_time = args.get("max_present_time", self.max_present_time)
        self.max_unconnected_time = args.get("max_unconnected_time", self.max_unconnected_time)
        self.min_repeate_time = args.get("min_repeate_time", self.min_repeate_time)
        if self.max_present_time is not None and self.max_unconnected_time is not None:
            if self.min_repeate_time is not None:
                if desc_str: desc_str += "\n"
                desc_str += u"Таймауты: %d-%d-%d" % (self.max_present_time, self.max_unconnected_time,
                                                     self.min_repeate_time)
        if desc_str:
            tooltip += "\n" + desc_str


        route_count = len(self.routes)
        if route_count > 0:
            desc_list = []
            for route in self.routes:
                r_desc = route.property('description')
                if r_desc is None or len(r_desc) == 0:
                    r_desc = route.id
                desc_list.append(r_desc)
            if route_count == 1:
                tooltip += u'\nМаршрут ' + desc_list[0]
            else:
                tooltip += u'\nМаршруты ' + u','.join(desc_list)

        self.vdtItem.setToolTip(tooltip)
        RoadCentre.gui.treeView.model().layoutChanged.emit()

    def updateIcon(self):
        if self.vdtItem.locked:
            self.setProperty('icon', QtGui.QPixmap("images/vdtl.png"))
        else:
            if self.vdtItem.state:
                self.setProperty('icon', QtGui.QPixmap("images/vdtp.png"))
            else:
                self.setProperty('icon', QtGui.QPixmap("images/vdt.png"))
        RoadCentre.gui.treeView.model().layoutChanged.emit()


class VdtMenu(SubMenu):
    def __init__(self):
        SubMenu.__init__(self, (u'Зона вызова', u'Зоны вызова'))
        a = self.menuMain.addAction(u'Показать на карте')
        a.connect('triggered()', self.showOnMap)
        self.selection = None

    def menu(self, obj):
        m = SubMenu.menu(self, obj)
        self.selection = obj
        # obj=obj[0]
        act_name = u"Разблокировать" if obj[0].vdtItem.locked else u"Заблокировать"
        act = m.addAction(act_name)
        act.connect('triggered()', self.setDtLocked)
        act_name = u"Деактивировать" if obj[0].vdtItem.state else u"Активировать"
        act = m.addAction(act_name)
        act.connect('triggered()', self.setDtActive)
        if len(obj) == 1:
            act = m.addAction(u"Свойства")
            act.connect('triggered()', self.vdtProperties)
        return m

    def setDtLocked(self):
        if not self.selection or len(self.selection) == 0:
            return
        to_lock = not self.selection[0].vdtItem.locked
        for obj in self.selection:
            RoadCentre.priority.setDtLocked(obj.id, to_lock)

    def setDtActive(self):
        if not self.selection or len(self.selection) == 0:
            return
        to_activate = not self.selection[0].vdtItem.state
        for obj in self.selection:
            RoadCentre.priority.setDtActive(obj.id, to_activate)

    def showOnMap(self):
        for o in RoadCentre.kernel.selection():
            if not hasattr(o, "className") or o.className() != "Vdetector":
                continue
            o.selected = True
            osmView.fitItem(o.vdtItem)

    def vdtProperties(self):
        if not self.selection or len(self.selection) != 1:
            return
        RoadCentre.priority.editVdt(self.selection[0])
